/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: OCF CardTerminalEvent sample.
 *
 */

package ocfevent;
import opencard.core.event.CTListener;
import opencard.core.event.CardTerminalEvent;
import opencard.core.event.EventGenerator;
import opencard.core.service.CardRequest;
import opencard.core.service.CardServiceException;
import opencard.core.service.SmartCard;
import opencard.core.terminal.CardTerminalException;
import opencard.core.terminal.CommandAPDU;
import opencard.core.terminal.ResponseAPDU;
import opencard.core.util.OpenCardPropertyLoadingException;
import opencard.opt.applet.BasicAppletCardService;
import jp.co.ricoh.dsdk.panel.AttributeType;
import jp.co.ricoh.dsdk.panel.Container;
import jp.co.ricoh.dsdk.panel.Font;
import jp.co.ricoh.dsdk.panel.Frame;
import jp.co.ricoh.dsdk.panel.LED;
import jp.co.ricoh.dsdk.panel.LEDExecutionException;
import jp.co.ricoh.dsdk.panel.Label;
import jp.co.ricoh.dsdk.panel.Text;
import jp.co.ricoh.dsdk.panel.Window;
import jp.co.ricoh.dsdk.panel.LED.Operation;
import jp.co.ricoh.dsdk.panel.LED.Type;
import jp.co.ricoh.dsdk.panel.event.KeyAdapter;
import jp.co.ricoh.dsdk.panel.event.KeyEvent;
import jp.co.ricoh.dsdk.xlet.UnavailableContainerException;
import jp.co.ricoh.dsdk.xlet.Xlet;
import jp.co.ricoh.dsdk.xlet.XletContext;
import jp.co.ricoh.dsdk.xlet.XletStateChangeException;

/**
 * OcfEvent
 * This is a sample application using the CardTerminalEvent of the OpenCard Framework.
 * 
 * OcfEvent
 * OpenCard Frameworkで、CardTerminalEventを利用するサンプルプログラムです。
 */
public class OcfEvent implements Xlet {
	
	/**
	 * the CommandAPDU to be sent to a JavaCard to obtain the token ID.
	 * 
	 * トークンIDを取得するためにカードに送信するコマンド
	 */
	private static final CommandAPDU commandAPDU = new CommandAPDU(new byte[]{
			(byte)0x80,(byte)0xCA,(byte)0x9F,(byte)0x7F,(byte)0x2D
	});
	
	/**
	 * The SmartCard object.
	 * 
	 * SmartCardオブジェクト
	 */
	private SmartCard card;
	
	/**
	 * The CTListener object.
	 * 
	 * CTListenerオブジェクト
	 */
	private CTListener listener;
	
	/**
	 * The root frame.
	 * 
	 * ルートフレーム
	 */
	private Frame frame;
	
	/**
	 * The label to be used to display a message.
	 * 
	 * メッセージ表示ラベル
	 */
	private Label msgLabel;
	
	/**
	 * The label to be used to display a token ID.
	 * 
	 * トークンID表示ラベル
	 */
	private Label tokenLabel;
	
	/**
	 * This is an initialization process.
	 * 1.obtains the root frame.
	 * 2.creates GUI.
	 * 3.sets LED state.
	 *  
	 * 初期化処理です。
	 * 1.ルートフレームを取得する。
	 * 2.GUIを作成する。
	 * 3.LEDを設定する。
	 */
	synchronized public void initXlet(XletContext xletContext) throws XletStateChangeException {
		try {
			Container parent = xletContext.getContainer();
			while (!(parent instanceof Frame)) {
				parent = parent.getParent();
				if (parent == null) break;
			}
			if (parent == null) {
				return;
			}
			frame = (Frame) parent;

			createGUI();
			
		} catch (UnavailableContainerException e) {
            throw new XletStateChangeException(e.getMessage());

        } catch (Exception e) {
            throw new XletStateChangeException(e.getMessage());
        }
        
        try {
			SmartCard.start();
			
		} catch (OpenCardPropertyLoadingException e) {
			setMessage(new Text("ERR_PROPERTYLOAD"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (CardServiceException e) {
			setMessage(new Text("ERR_CARDSERVICE"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (CardTerminalException e) {
			setMessage(new Text("ERR_CARDTERMINAL"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (ClassNotFoundException e) {
			setMessage(new Text("ERR_CLASSNOTFOUND"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (Throwable t) {
			setMessage(new Text("ERR_THROWABLE"));
			t.printStackTrace();
			throw new XletStateChangeException(t.getMessage());
		}
				
        setLED(Type.START, Operation.GRN_ON);
	}

	/**
	 * Activation
	 * 
	 * 活性化。
	 */
	synchronized public void startXlet() throws XletStateChangeException {
	}
	
	/**
	 * Stop.
	 * 
	 * 停止。
	 */
	synchronized public void pauseXlet() {
	}

	/**
	 * End process.
	 * This process is similar to the process followed after the clear/stop key is pressed.
	 * 
	 * 終了処理。
	 * クリアストップキー押された時と同様の処理を行う。
	 */
	synchronized public void destroyXlet(boolean destroy) throws XletStateChangeException {
		pushClearStopKey();
		
		try {
			SmartCard.shutdown();
			
		} catch (CardTerminalException e) {
			setMessage(new Text("ERR_CARDTERMINAL"));
			e.printStackTrace();
		}
	}
	
	/**
	 * GUI creation.
	 * This is called from the initXlet method, and creates GUI on the operation panel.
	 * 1.creates a title window.
	 * 2.creates a title label.
	 * 3.creates a message label.
	 * 4.creates a token ID label.
	 * 5.registers a KeyListener to the root frame for processing KeyEvents. 
	 * 
	 * GUI作成。
	 * initXletメソッドから呼び出され、オペパネにＧＵＩを作成します。
	 * 1.タイトル表示ウィンドウを作成する。
	 * 2.タイトル表示ラベルを作成する。
	 * 3.メッセージ表示ラベルを作成する。
	 * 4.トークンID表示ラベルを作成する。
	 * 5.キーリスナーを登録して、キーイベントを処理する。
	 */
	private void createGUI() {
		Window titleWindow = new Window(frame);
		titleWindow.setAttributeType(AttributeType.INVERSE);
		titleWindow.setLocation(0, 0);
		titleWindow.setSize(frame.getWidth(), 32);

		Label titleLabel = new Label(new Text("APP_TITLE"));  
		titleLabel.setFont(Font.F16);
		titleLabel.setAttributeType(AttributeType.INVERSE);
		titleLabel.setLocation(0, 0);
		titleLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(titleLabel);
		
		msgLabel = new Label(new Text("MSG_PUSHSTART"));  
		msgLabel.setFont(Font.F16);
		msgLabel.setAttributeType(AttributeType.INVERSE);
		msgLabel.setLocation(0, 16);
		msgLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(msgLabel);

		tokenLabel = new Label();
		tokenLabel.setSize(titleWindow.getWidth()-200, 16);
		tokenLabel.setLocation(
				100 ,
				(frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 2);
		frame.add(tokenLabel);
		setToken(null);
		
		frame.addKeyListener(
				new KeyAdapter() {
					public void keyPressed(KeyEvent keyEvent) {
						switch (keyEvent.getKeyCode()) {
						case KeyEvent.KEY_START:     // Start key
							pushStartKey();	
							break;
						case KeyEvent.KEY_CLR_STOP:  // Clear/Stop key
							pushClearStopKey();
							break;
						}
					}
				}
			);
	}
	
	/**
	 * The process followed after the start key is pressed.
	 * 1.registers a CTListener to the EventGenerator.
	 * 2.calls createEventsForPresentCards method because it is likely that 
	 *   a card already exists in the card reader.
	 * 3.sets LED state.
	 * 
	 * スタートキーが、押された時の処理
	 * 1.リスナを登録する。
	 * 2.リスナ登録前からカードがカードリーダに挿入されていた可能性があるので、
	 *   EventGeneratorのcreateEventsForPresentCardsメソッドを呼び出す。
	 * 3.LEDを設定する。
	 */
	synchronized private void pushStartKey() {
		if (listener == null) {
			setToken(null);
			
			setMessage(new Text("MSG_SETCARD"));
			
			CTListener listener = new CTListener() {
				
				/**
				 * The process followed after a card is inserted to the card reader. 
				 * 1.obtains a SmartCard object.
				 * 2.obtains a CardService object.
				 * 3.obtains a mutex lock of the card.
				 * 4.obtains token ID.
				 * 5.displays token ID.
				 * 6.releases the mutex lock of the card.
				 * 7.sets LED state.
				 * 
				 * カードが、認識された時の処理
				 * 1.カードを取得する。
				 * 2.カードサービスを取得する。
				 * 3.カードをロックする。
				 * 4.トークンIDを取得する。
				 * 5.トークンIDを表示する。
				 * 6.カードのロックを解除する。
				 * 7.LEDを設定する。
				 */
				public void cardInserted(CardTerminalEvent event) throws CardTerminalException {
					try {
						card = SmartCard.getSmartCard (event,
								new CardRequest(CardRequest.ANYCARD, null, null));
						
						if (card != null) {
							BasicAppletCardService appletService =
								(BasicAppletCardService)card.getCardService(
										BasicAppletCardService.class, false);
							
							ResponseAPDU responseAPDU = 
								appletService.sendCommandAPDU(null, commandAPDU);
							
							byte[] data = responseAPDU.data();
							
							byte[] tokenID = new byte[10];
							System.arraycopy(data, 0x03, tokenID, 0, 2);
							System.arraycopy(data, 0x05, tokenID, 2, 2);
							System.arraycopy(data, 0x13, tokenID, 4, 2);
							System.arraycopy(data, 0x0F, tokenID, 6, 4);
							
							setToken(tokenID);
							
							setMessage(new Text("MSG_SUCCESS_INSERTED"));							
						}
						
					} catch (CardServiceException e) {
						setMessage(new Text("ERR_CARDSERVICE"));
						e.printStackTrace();
						
					} catch (ClassNotFoundException e) {
						setMessage(new Text("ERR_CLASSNOTFOUND"));
						e.printStackTrace();
						
					} catch (Throwable t) {
						setMessage(new Text("ERR_THROWABLE"));
						t.printStackTrace();
						
					}
					
					setLED(Type.START, Operation.GRN_BLINK);
				}
				
				/**
				 * The process followed after a card is removed from the card reader.
				 * 1.closes the SmartCard object.
				 * 2.displays a message that indicates the removal of a card.
				 * 3.sets LED state.
				 * 
				 * カードが、外された時の処理
				 * 1.カードを破棄する。
				 * 2.カードが外された旨を表示する。
				 * 3.LEDを設定する。
				 */
				public void cardRemoved(CardTerminalEvent event) throws CardTerminalException {
					if (card != null) {
						card.close();
						card = null;
					}
					
					setToken(null);
					setMessage(new Text("MSG_SUCCESS_REMOVED"));
					
					setLED(Type.START, Operation.RED_BLINK);
				}
				
			};
			
			EventGenerator.getGenerator().addCTListener(listener);
			
			this.listener = listener;
			
			try {
				EventGenerator.getGenerator().createEventsForPresentCards(listener);
				
			} catch (CardTerminalException e) {
				setMessage(new Text("ERR_CARDTERMINAL"));
				e.printStackTrace();
			}

			setLED(Type.START, Operation.RED_BLINK);
		}
		
	}
	
	/**
	 * The process followed after the claer/stop key is pressed.
	 * 1.removes the CTListener from the EventGenerator.
	 * 2.closes the SmartCard object.
	 * 3.sets LED state.
	 * 
	 * クリア/ストップキーが、押された時の処理
	 * 1.リスナを削除する。
	 * 2.カードを破棄する。
	 * 3.LEDを設定する。
	 */
	synchronized private void pushClearStopKey() {
		if (listener != null) {
			
			EventGenerator.getGenerator().removeCTListener(listener);
			listener = null;
			
			if (card != null) {
				try {
					card.close();
					
				} catch (CardTerminalException e) {
					setMessage(new Text("ERR_CARDTERMINAL"));
					e.printStackTrace();
				}
				
				card = null;
			}
			
			setMessage(new Text("ERR_ABORT"));
			
			setLED(Type.START, Operation.GRN_ON);
		}
	}
	
	/**
	 * Displays the given token ID on the token ID label.
	 * 
	 * トークンIDを、トークンID表示ラベルに表示する。
	 */
	private void setToken(byte[] tokenID) {
		final String replace = "NUMBER";
		Text text = new Text("TOKENID_LABEL");
		
		if (tokenID == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < tokenID.length; i++) {
				n = tokenID[i];
				n &=0x000000ff;
				if (n<=0x0000000f) {
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		tokenLabel.setText(text);
		tokenLabel.repaint();
	}
	
	/**
	 * Displays the given message on the message label.
	 * 
	 * メッセージを、メッセージ表示ラベルに表示する。
	 */
	private void setMessage(Text message) {
		msgLabel.setText(message);
		msgLabel.repaint();
	}
	
	/**
	 * Activates LED with the given Type and the given Operation.
	 * 
	 * LEDを動作させる。
	 */
	private void setLED(Type type, Operation ope) {
		try {
			new LED(frame, type, ope).blink();
		} catch (LEDExecutionException e) {
			e.printStackTrace();
		}
	}
}
